// Compressor
// Josh Wade
// 11/6/2018
// Part one of a two part assignment
// Reads in data from a list of vertices
// Compresses those values and writes out the results as a bit stream

#include <stdio.h>
#include <stdlib.h>
#include <float.h>
#include <math.h>
#include <string.h>

// Prototypes
int createFileNames();
int openInputFile();
int openOutputFiles();
int readArguments(int argc, char *argv[]);
int readFileData();
int compressToBits();
unsigned short storeInShort(unsigned short spaceTaken, unsigned short bucketValue);
void print16Bit(FILE * file, int input);
float getMax(float a, float b);
float getMin(float a, float b);
int buildBuckets();
unsigned short assignBucket(float number);

// Sizes for different data types
unsigned short charLength = 8;
unsigned short shortLength = 16;

// Data for constructing file names
char *inputFileName;
char fullBinFileName[30];
char fullVertFileName[30];
char binaryFileName[] = "binary";
char vertsFileName[] = "vertsCleaned";
char binaryExtension[] = ".bin";
char textExtension[] = ".txt";

// File pointers 
FILE *outputBinFile = NULL;
FILE *outputVertFile = NULL;
FILE *inputFile = NULL;

// How many bits are in our compressed data points
unsigned short bitCount;

// Min and max values of the vertices
float maxValue = FLT_MIN;
float minValue = FLT_MAX;

// How big each "bucket" of data is
float bucketSize = 0;

// How many "buckets" there are 
unsigned long bucketCount = 0;

// Our temporary storage for the short we write out
unsigned short writeBuffer = 0;

int main(int argc, char *argv[])
{
	// Run our functions, checking for errors at every step
	if (readArguments(argc, argv))
		return 1;

	if (createFileNames())
		return 1;

	if (openInputFile())
		return 1;

	if (openOutputFiles())
		return 1;

	if (readFileData())
		return 1;

	if (buildBuckets())
		return 1;

	// Reopen the input file to restart and reread
	if (openInputFile())
		return 1;

	if (compressToBits())
		return 1;

	// Close the files 
	fclose(outputBinFile);
	fclose(outputVertFile);
	fclose(inputFile);

	return 0;
}

// Creates the file names based on our bit counts
int createFileNames()
{
	// Parse our bit count as a string
	char bitCountAsString[3];
	_itoa(bitCount, bitCountAsString, 10);

	// Build the bin file name with the bit count in it
	strcat(fullBinFileName, binaryFileName);
	strcat(fullBinFileName, bitCountAsString);
	strcat(fullBinFileName, binaryExtension);

	// Build the cleaned vert file name
	// This originally contained the bitCount, but this has been considered redundant
	strcat(fullVertFileName, vertsFileName);
	strcat(fullVertFileName, textExtension);

	return 0;
}

// Opens the input file 
int openInputFile()
{
	// Check if the file correctly opened
	inputFile = fopen(inputFileName, "r");
	if (inputFile == NULL)
	{
		printf("Problems encountered opening file (%s)! Please try again.", inputFileName);
		getchar();
		return 1;
	}

	return 0;
}

// Opens the output files
int openOutputFiles()
{
	// Check if each file correctly opened
	outputBinFile = fopen(fullBinFileName, "wb");
	outputVertFile = fopen(fullVertFileName, "w");
	if (outputBinFile == NULL)
	{
		printf("Problems encountered opening file (%s)! Please try again.", fullBinFileName);
		getchar();
		return 1;
	}
	if (outputVertFile == NULL)
	{
		printf("Problems encountered opening file (%s)! Please try again.", fullVertFileName);
		getchar();
		return 1;
	}

	return 0;
}

// Read in command line arguments
int readArguments(int argc, char *argv[])
{
	// Check if the argument count is correct
	if (argc != 3) {
		printf("Invalid argument list! Please try again.");
		getchar();
		return 1;
	}

	// Read in file name & bit count
	bitCount = (unsigned short)atoi(argv[1]);
	inputFileName = argv[2];

	bucketCount = (unsigned long)(pow(2, bitCount));

	return 0;
}

// Passes over the data to figure out the min value, max value, and total number of verts
int readFileData()
{
	int index = 0;
	float x, y, z;
	char line[256];

	while (fgets(line, sizeof(line), inputFile))
	{
		// Scan the x y z from our current line
		if (sscanf(line, "%d: %f %f %f", &index, &x, &y, &z) != 4)
		{
			printf("Problems encountered reading input file (%s)! Please try again.", fullVertFileName);
			getchar();
			return 1;
		}

		// Compare that with our mins and maxes
		maxValue = getMax(x, maxValue);
		maxValue = getMax(y, maxValue);
		maxValue = getMax(z, maxValue);

		minValue = getMin(x, minValue);
		minValue = getMin(y, minValue);
		minValue = getMin(z, minValue);
	}

	// Save these values to the bin file
	fprintf(outputBinFile, "%.5f %.5f %d\n", minValue, maxValue, (index + 1));

	return 0;
}

// Compresses every float type to a bucket index and writes it out to the output bin file
int compressToBits()
{
	int index = 0;
	float x, y, z;
	char line[256];
	unsigned short spaceTaken = 0;

	// Read current triplet of numbers
	while (fgets(line, sizeof(line), inputFile))
	{
		sscanf(line, "%d: %f %f %f", &index, &x, &y, &z);

		// Store each of these into our toWrite short
		spaceTaken = storeInShort(spaceTaken, assignBucket(x));
		spaceTaken = storeInShort(spaceTaken, assignBucket(y));
		spaceTaken = storeInShort(spaceTaken, assignBucket(z));

		fprintf(outputVertFile, "%.5f\n%.5f\n%.5f\n", x, y, z);
	}

	// Slide to the top edge when we get to the end, then print it out
	writeBuffer = writeBuffer << (shortLength - spaceTaken);
	print16Bit(outputBinFile, writeBuffer);

	return 0;
}

// Store the given short and try to stuff it in our short toWrite 
unsigned short storeInShort(unsigned short spaceTaken, unsigned short bucketValue)
{
	// Check how much space we have left 
	if ((spaceTaken + bitCount) < shortLength)
	{
		// If there's enough space, shift over toWrite and write bucketValue in there
		writeBuffer = writeBuffer << bitCount;
		writeBuffer = writeBuffer | bucketValue;
		// Then increment spaceTaken
		spaceTaken += bitCount;
	}
	else
	{
		// If there is is no space left, shift in whatever we can 
		writeBuffer = writeBuffer << (shortLength - spaceTaken);

		// Split our data out to find what can make it into the buffer
		unsigned short cutoff = (bitCount - (shortLength - spaceTaken));
		unsigned short bottomBucketValue = bucketValue;
		unsigned short topBucketValue = bucketValue;

		// Shift items around to cut data off into a top and bottom section
		bottomBucketValue = bottomBucketValue << (shortLength - cutoff);
		bottomBucketValue = bottomBucketValue >> (shortLength - cutoff);
		topBucketValue = topBucketValue >> cutoff;

		// Fit as much of the bucketValue as possible, then print it out
		writeBuffer = writeBuffer | topBucketValue;
		print16Bit(outputBinFile, writeBuffer);
		//printf("%hu\n", toWrite);

		// Write out that short, then clear it, shifting in whatever you didn't have space 
		writeBuffer = 0 | bottomBucketValue;

		// Then, increment the spaceTaken
		spaceTaken += bitCount;
		spaceTaken %= shortLength;

	}

	return spaceTaken;
}

// Print 16 bit number to file 
void print16Bit(FILE * file, int input)
{
	fprintf(file, "%c", input);
	fprintf(file, "%c", input >> charLength);
}

// Get the maximum of two floats
float getMax(float a, float b)
{
	if (a > b)
		return a;
	return b;
}

// Get the minimum of two floats
float getMin(float a, float b)
{
	if (a < b)
		return a;
	return b;
}

// Figure out the sizes of our buckets 
int buildBuckets()
{
	// Warn if we're attempting a by-0 division 
	if (bucketCount < 1)
	{
		printf("Problems encountered creating bucket values! Please try a smaller bitcount.");
		getchar();
		return 1;
	}

	// Determine the increment size for one bucket to the next 
	bucketSize = (float(maxValue - minValue)) / (bucketCount);
	return 0;
}

// Figures out the bucket number for this float value
unsigned short assignBucket(float number)
{
	unsigned long i = 0;
	for (i = 0; i < bucketCount; i++) {
		float lowerBucketEdge = minValue + (i * bucketSize);
		if (number < lowerBucketEdge) {
			return (unsigned short)(i - 1);
		}
	}

	return (unsigned short)(bucketCount - 1);
}